//
//  FairDircDet.cxx
//  
//
//  Created by Kunnawalkam Raghav on 3/1/12.
//  Copyright (c) 2012 Stony Brook University. All rights reserved.
//


#include "FairDircDet.h"

#include "FairDircDetPoint.h"
#include "FairDircDetGeo.h"
#include "FairDircDetGeoPar.h"

#include "FairVolume.h"
#include "FairGeoVolume.h"
#include "FairGeoNode.h"
#include "FairRootManager.h"
#include "FairGeoLoader.h"
#include "FairGeoInterface.h"
#include "FairRun.h"
#include "FairRuntimeDb.h"
#include "FairDetectorList.h"
#include "FairStack.h"

#include "TClonesArray.h"
#include "TVirtualMC.h"

#include <iostream>
using std::cout;
using std::endl;

// I am going to use kFALSE instead of kTRUE just for now to build the detector. Later we can use kTRUE to call ProcessHits and do physics. 
DetectorId kFairDircDet;

FairDircDet::FairDircDet()
: FairDetector("FairDircDet", kFALSE, kFairDircDet), 
fTrackID(-1),
fVolumeID(-1),
fPos(),
fMom(),
fTime(-1.),
fLength(-1.),
fELoss(-1),
fFairDircDetPointCollection(new TClonesArray("FairDircDetPoint"))
{
}

FairDircDet::FairDircDet(const char* name, Bool_t active)
: FairDetector(name, active, kFairDircDet),
fTrackID(-1),
fVolumeID(-1),
fPos(),
fMom(),
fTime(-1.),
fLength(-1.),
fELoss(-1),
fFairDircDetPointCollection(new TClonesArray("FairDircDetPoint"))
{
}

FairDircDet::~FairDircDet()
{
    if (fFairDircDetPointCollection) {
        fFairDircDetPointCollection->Delete();
        delete fFairDircDetPointCollection;
    }
}

void FairDircDet::Initialize()
{
    FairDetector::Initialize();
    FairRuntimeDb* rtdb= FairRun::Instance()->GetRuntimeDb();
    FairDircDetGeoPar* par=(FairDircDetGeoPar*)(rtdb->getContainer("FairDircDetGeoPar"));
}

Bool_t  FairDircDet::ProcessHits(FairVolume* vol)
{
    /** This method is called from the MC stepping */
    
    //Set parameters at entrance of volume. Reset ELoss.
    if ( gMC->IsTrackEntering() ) {
        fELoss  = 0.;
        fTime   = gMC->TrackTime() * 1.0e09;
        fLength = gMC->TrackLength();
        gMC->TrackPosition(fPos);
        gMC->TrackMomentum(fMom);
    }
    
    // Sum energy loss for all steps in the active volume
    fELoss += gMC->Edep();
    
    // Create FairDircDetPoint at exit of active volume
    if ( gMC->IsTrackExiting()    ||
        gMC->IsTrackStop()       ||
        gMC->IsTrackDisappeared()   ) {
        fTrackID  = gMC->GetStack()->GetCurrentTrackNumber();
        fVolumeID = vol->getMCid();
        if (fELoss == 0. ) { return kFALSE; }
        AddHit(fTrackID, fVolumeID, TVector3(fPos.X(),  fPos.Y(),  fPos.Z()),
               TVector3(fMom.Px(), fMom.Py(), fMom.Pz()), fTime, fLength,
               fELoss);
        
        // Increment number of FairDircDet points in TParticle
        FairStack* stack = (FairStack*) gMC->GetStack();
        stack->AddPoint(kFairDircDet);
    }
    
    return kTRUE;
}

void FairDircDet::EndOfEvent()
{
    
    fFairDircDetPointCollection->Clear();
    
}



void FairDircDet::Register()
{
    
    /** This will create a branch in the output tree called
     FairDircDetPoint, setting the last parameter to kFALSE means:
     this collection will not be written to the file, it will exist
     only during the simulation.
     */
    
    FairRootManager::Instance()->Register("FairDircDetPoint", "FairDircDet",
                                          fFairDircDetPointCollection, kTRUE);
    
}


TClonesArray* FairDircDet::GetCollection(Int_t iColl) const
{
    if (iColl == 0) { return fFairDircDetPointCollection; }
    else { return NULL; }
}

void FairDircDet::Reset()
{
    fFairDircDetPointCollection->Clear();
}

void FairDircDet::ConstructGeometry()
{
    /** If you are using the standard ASCII input for the geometry
     just copy this and use it for your detector, otherwise you can
     implement here you own way of constructing the geometry. */
    
    FairGeoLoader*    geoLoad = FairGeoLoader::Instance();
    FairGeoInterface* geoFace = geoLoad->getGeoInterface();
    FairDircDetGeo*  Geo  = new FairDircDetGeo();
    Geo->setGeomFile(GetGeometryFileName());
    geoFace->addGeoModule(Geo);
    
    Bool_t rc = geoFace->readSet(Geo);
    if (rc) { Geo->create(geoLoad->getGeoBuilder()); }
    TList* volList = Geo->getListOfVolumes();
    
    // store geo parameter
    FairRun* fRun = FairRun::Instance();
    FairRuntimeDb* rtdb= FairRun::Instance()->GetRuntimeDb();
    FairDircDetGeoPar* par=(FairDircDetGeoPar*)(rtdb->getContainer("FairDircDetGeoPar"));
    TObjArray* fSensNodes = par->GetGeoSensitiveNodes();
    TObjArray* fPassNodes = par->GetGeoPassiveNodes();
    
    TListIter iter(volList);
    FairGeoNode* node   = NULL;
    FairGeoVolume* aVol=NULL;
    
    while( (node = (FairGeoNode*)iter.Next()) ) {
        aVol = dynamic_cast<FairGeoVolume*> ( node );
        if ( node->isSensitive()  ) {
            fSensNodes->AddLast( aVol );
        } else {
            fPassNodes->AddLast( aVol );
        }
    }
    par->setChanged();
    par->setInputVersion(fRun->GetRunId(),1);
    
    ProcessNodes ( volList );
}

FairDircDetPoint* FairDircDet::AddHit(Int_t trackID, Int_t detID,
                                            TVector3 pos, TVector3 mom,
                                            Double_t time, Double_t length,
                                            Double_t eLoss)
{
    TClonesArray& clref = *fFairDircDetPointCollection;
    Int_t size = clref.GetEntriesFast();
    return new(clref[size]) FairDircDetPoint(trackID, detID, pos, mom,
                                                time, length, eLoss, pos.Mag(), pos.Phi(), pos.Theta());
}

ClassImp(FairDircDet)
