//
//  FairSimpletracker.cxx
//  
//
//  Created by Kunnawalkam Raghav on 2/25/12.
//  Copyright (c) 2012 Stony Brook University. All rights reserved.
//

#include "FairSimpletracker.h"

#include "FairSimpletrackerPoint.h"
#include "FairSimpletrackerGeo.h"
#include "FairSimpletrackerGeoPar.h"

#include "FairVolume.h"
#include "FairGeoVolume.h"
#include "FairGeoNode.h"
#include "FairRootManager.h"
#include "FairGeoLoader.h"
#include "FairGeoInterface.h"
#include "FairRun.h"
#include "FairRuntimeDb.h"
#include "FairDetectorList.h"
#include "FairStack.h"

#include "TClonesArray.h"
#include "TVirtualMC.h"

#include <iostream>
using std::cout;
using std::endl;

DetectorId kFairSimpletracker;

FairSimpletracker::FairSimpletracker()
  : FairDetector("FairSimpletracker", kTRUE, kFairSimpletracker),
    fTrackID(-1),
    fVolumeID(-1),
    fPos(),
    fMom(),
    fTime(-1.),
    fLength(-1.),
    fELoss(-1),
    fFairSimpletrackerPointCollection(new TClonesArray("FairSimpletrackerPoint"))
{
}

FairSimpletracker::FairSimpletracker(const char* name, Bool_t active)
  : FairDetector(name, active, kFairSimpletracker),
    fTrackID(-1),
    fVolumeID(-1),
    fPos(),
    fMom(),
    fTime(-1.),
    fLength(-1.),
    fELoss(-1),
    fFairSimpletrackerPointCollection(new TClonesArray("FairSimpletrackerPoint"))
{
}

FairSimpletracker::~FairSimpletracker()
{
  if (fFairSimpletrackerPointCollection) {
    fFairSimpletrackerPointCollection->Delete();
    delete fFairSimpletrackerPointCollection;
  }
}

void FairSimpletracker::Initialize()
{
  FairDetector::Initialize();
  FairRuntimeDb* rtdb= FairRun::Instance()->GetRuntimeDb();
  FairSimpletrackerGeoPar* par=(FairSimpletrackerGeoPar*)(rtdb->getContainer("FairSimpletrackerGeoPar"));
}

Bool_t  FairSimpletracker::ProcessHits(FairVolume* vol)
{
  /** This method is called from the MC stepping */

  //Set parameters at entrance of volume. Reset ELoss.
  if ( gMC->IsTrackEntering() ) {
    fELoss  = 0.;
    fTime   = gMC->TrackTime() * 1.0e09;
    fLength = gMC->TrackLength();
    gMC->TrackPosition(fPos);
    gMC->TrackMomentum(fMom);
  }

  // Sum energy loss for all steps in the active volume
  fELoss += gMC->Edep();

  // Create FairSimpletrackerPoint at exit of active volume
  if ( gMC->IsTrackExiting()    ||
       gMC->IsTrackStop()       ||
       gMC->IsTrackDisappeared()   ) {
    fTrackID  = gMC->GetStack()->GetCurrentTrackNumber();
    fVolumeID = vol->getMCid();
    if (fELoss == 0. ) { return kFALSE; }
    AddHit(fTrackID, fVolumeID, TVector3(fPos.X(),  fPos.Y(),  fPos.Z()),
           TVector3(fMom.Px(), fMom.Py(), fMom.Pz()), fTime, fLength,
           fELoss);

    // Increment number of FairSimpletracker points in TParticle
    FairStack* stack = (FairStack*) gMC->GetStack();
    stack->AddPoint(kFairSimpletracker);
  }

  return kTRUE;
}

void FairSimpletracker::EndOfEvent()
{

  fFairSimpletrackerPointCollection->Clear();

}



void FairSimpletracker::Register()
{

  /** This will create a branch in the output tree called
      FairSimpletrackerPoint, setting the last parameter to kFALSE means:
      this collection will not be written to the file, it will exist
      only during the simulation.
  */

  FairRootManager::Instance()->Register("FairSimpletrackerPoint", "FairSimpletracker",
                                        fFairSimpletrackerPointCollection, kTRUE);

}


TClonesArray* FairSimpletracker::GetCollection(Int_t iColl) const
{
  if (iColl == 0) { return fFairSimpletrackerPointCollection; }
  else { return NULL; }
}

void FairSimpletracker::Reset()
{
  fFairSimpletrackerPointCollection->Clear();
}

void FairSimpletracker::ConstructGeometry()
{
  /** If you are using the standard ASCII input for the geometry
      just copy this and use it for your detector, otherwise you can
      implement here you own way of constructing the geometry. */

  FairGeoLoader*    geoLoad = FairGeoLoader::Instance();
  FairGeoInterface* geoFace = geoLoad->getGeoInterface();
  FairSimpletrackerGeo*  Geo  = new FairSimpletrackerGeo();
  Geo->setGeomFile(GetGeometryFileName());
  geoFace->addGeoModule(Geo);

  Bool_t rc = geoFace->readSet(Geo);
  if (rc) { Geo->create(geoLoad->getGeoBuilder()); }
  TList* volList = Geo->getListOfVolumes();

  // store geo parameter
  FairRun* fRun = FairRun::Instance();
  FairRuntimeDb* rtdb= FairRun::Instance()->GetRuntimeDb();
  FairSimpletrackerGeoPar* par=(FairSimpletrackerGeoPar*)(rtdb->getContainer("FairSimpletrackerGeoPar"));
  TObjArray* fSensNodes = par->GetGeoSensitiveNodes();
  TObjArray* fPassNodes = par->GetGeoPassiveNodes();

  TListIter iter(volList);
  FairGeoNode* node   = NULL;
  FairGeoVolume* aVol=NULL;

  while( (node = (FairGeoNode*)iter.Next()) ) {
    aVol = dynamic_cast<FairGeoVolume*> ( node );
    if ( node->isSensitive()  ) {
      fSensNodes->AddLast( aVol );
    } else {
      fPassNodes->AddLast( aVol );
    }
  }
  par->setChanged();
  par->setInputVersion(fRun->GetRunId(),1);

  ProcessNodes ( volList );
}

FairSimpletrackerPoint* FairSimpletracker::AddHit(Int_t trackID, Int_t detID,
    TVector3 pos, TVector3 mom,
    Double_t time, Double_t length,
    Double_t eLoss)
{
  TClonesArray& clref = *fFairSimpletrackerPointCollection;
  Int_t size = clref.GetEntriesFast();
  return new(clref[size]) FairSimpletrackerPoint(trackID, detID, pos, mom,
         time, length, eLoss, pos.Mag(), pos.Phi(), pos.Theta());
}

ClassImp(FairSimpletracker)
